package Command;

use strict;
use warnings;

# Adapted from http://www.mattsnotes.com/perl/my-perl-module-template
our $VERSION = '0.0.0.2'; # VERSION
 
use File::Basename qw(basename);
use Log::Log4perl qw(get_logger);
use POSIX qw(strftime);

use Carp ();

sub new {
  my $class = shift;
 
  my ($instance) = (@_);
  my $self = {};
  bless $self, $class;
 
  $self->{instance} = $instance;
  $self->{connection} = 0;
  $self->{progname} = basename($0);
  $self->{progname} =~ s/\..*$//;
  $self->{log_filename} = $self->log_filename();
  $self->{logger} = $self->_init_logging();
  $self->{binPath} = "";
  $self->{workDir} = "";
  $self->{homeDir} = "";
  $self->{vpatch_to_process}   = "";
  $self->{expected_antecedents}  = "";
  $self->{expected_descendants}  = "";
  $self->{expected_signatures}   = "";
  $self->{expected_source_files} = "";

  return $self;
}
 
sub log_filename {
  my $self = shift;
 
  if ( ! $self->{log_filename}) {
 
    my ($log_path, $log_file);

    if ($self->{instance}) {
      $log_path = "logs/$self->{instance}/logs/".$self->{progname};
      $log_file = "$self->{instance}.".$self->{progname}.".".timestamp().".$$.log";
    } else {
      $log_path = "logs/".$self->{progname};
      $log_file = $self->{progname}.".".timestamp().".$$.log";
    }
 
    $self->{log_path} = $log_path;
    $self->{log_file} = $log_file;
    $self->{log_filename} = "$log_path/$log_file";
  }

  return $self->{log_path}."/".$self->{log_file};
}
 
sub _init_logging {
  my $self = shift;
 
  my $log_path = $self->{log_path};
  my $log_file = $self->{log_file};
 
  mkdir "logs" unless -e "logs";
  mkdir $log_path unless -e $log_path;
 
  my $conf = "
    log4perl.logger                                      = DEBUG, FileApp, ScreenApp
    log4perl.appender.FileApp                            = Log::Log4perl::Appender::File
    log4perl.appender.FileApp.filename                   = $log_path/$log_file
    log4perl.appender.FileApp.layout                     = PatternLayout
    log4perl.appender.FileApp.layout.ConversionPattern   = %d %p %m%n
    log4perl.appender.ScreenApp                          = Log::Log4perl::Appender::Screen
    log4perl.appender.ScreenApp.layout                   = PatternLayout
    log4perl.appender.ScreenApp.layout.ConversionPattern = %d %p %m%n
  ";
  Log::Log4perl->init( \$conf );
  print "Logging messages to $log_path/$log_file\n";

  return get_logger();
}
 
sub progname {
  my $self = shift;
  return $self->{progname};
}
 
sub logger {
  my $self = shift;
  return $self->{logger};
}
 
sub _initialise_logging {
  my $self = shift;
  $self->{logger}->info("Started");

  return;
}
 
sub timestamp {
  return strftime "%Y-%m-%d_%H%M%S", localtime;
}
 
sub setBinPath {
  my $self = shift;
  my ($path) = @_;
  $self->{binPath} = $path;
}

sub getBinPath { 
  my $self = shift;
  return $self->{binPath};
}

sub setWorkDir {
  my $self = shift;
  my ($dir) = @_;
  $self->{workDir} = $dir;
}  

sub getWorkDir {
  my $self = shift;
  return $self->{workDir};
}

sub setHomeDir { 
  my $self = shift;
  my ($dir) = @_;
  $self->{homeDir} = $dir;
}

sub getHomeDir { 
  my $self = shift;
  return $self->{homeDir};
}

sub setVpatchToProcess {
  my $self = shift;
  my ($vpatch) = @_;
  $self->{vpatch_to_process} = $vpatch;  
}

sub getVpatchToProcess {
  my $self = shift;
  return $self->{vpatch_to_process};
}

sub setExpectedAntecedents { 
  my $self = shift;
  my ($expectedAntecedents) = @_;
  $self->{expected_antecedents} = $expectedAntecedents;  
}

sub getExpectedAntecedents { 
  my $self = shift;
  return $self->{expected_antecedents};
}

sub setExpectedDescendants {
  my $self = shift;
  my ($expectedDescendants) = @_;
  $self->{expected_descendants} = $expectedDescendants;
}

sub getExpectedDescendants {
  my $self = shift;
  return $self->{expected_descendants};
}

sub setExpectedSignatures {
  my $self = shift;
  my ($expectedSignatures) = @_;
  $self->{expected_signatures} = $expectedSignatures;
}

sub getExpectedSignatures {
  my $self = shift;
  return $self->{expected_signatures};
}

sub setExpectedSourceFiles {
  my $self = shift;
  my ($expectedSourceFiles) = @_;
  $self->{expected_source_files} = $expectedSourceFiles;
}

sub getExpectedSourceFiles {
  my $self = shift;
  return $self->{expected_source_files};
}

sub destroy {
  my $self = shift;
  $self->{logger}->info("Finished");
}
 
1;
 
__END__
